<?php
/**
 * Ninja Saga AMF Gateway dengan SabreAMF - Fixed Version
 * 
 * Fix untuk kompatibilitas dengan ZendAMF client
 */

require_once 'vendor/autoload.php';
require_once 'config.php';
require_once 'db.php';

use SabreAMF\AMF;
use SabreAMF\AMF3\Deserializer;
use SabreAMF\AMF3\Serializer;

// Set headers
header('Content-Type: application/x-amf');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Include service classes
require_once 'services/SystemService.php';
require_once 'services/CharacterDAO.php';
require_once 'services/SystemData.php';

// AMF Handler
class AMFHandler {
    private $db;
    
    public function __construct($db) {
        $this->db = $db;
    }
    
    public function handle($service, $method, $params) {
        $serviceClass = $this->getServiceClass($service);
        
        if (!$serviceClass) {
            throw new Exception("Service not found: $service");
        }
        
        if (!method_exists($serviceClass, $method)) {
            throw new Exception("Method not found: $service.$method");
        }
        
        return call_user_func_array([$serviceClass, $method], $params);
    }
    
    private function getServiceClass($service) {
        $services = [
            'SystemService' => 'SystemService',
            'CharacterDAO' => 'CharacterDAO',
            'SystemData' => 'SystemData',
        ];
        
        if (!isset($services[$service])) {
            return null;
        }
        
        $className = $services[$service];
        return new $className($this->db);
    }
}

// Main handler
try {
    $db = getDBConnection();
    $handler = new AMFHandler($db);
    
    // Read input
    $input = file_get_contents('php://input');
    
    // Log for debugging
    error_log("=== AMF Request Debug ===");
    error_log("Input length: " . strlen($input) . " bytes");
    if (strlen($input) > 0) {
        error_log("First 20 bytes (hex): " . bin2hex(substr($input, 0, min(20, strlen($input)))));
    }
    
    // Initialize AMF
    $amf = new AMF();
    $amf->setDeserializer(new Deserializer());
    $amf->setSerializer(new Serializer());
    
    // Parse request
    $service = 'SystemService';
    $method = 'login';
    $params = ['test', 'test123', BUILD_NO];
    
    if (strlen($input) > 0) {
        try {
            $request = $amf->deserialize($input);
            
            // SabreAMF returns different structures depending on version
            if (is_object($request)) {
                // Check for Message structure
                if (isset($request->target)) {
                    $service = $request->target;
                    $method = isset($request->operation) ? $request->operation : 'login';
                    $params = isset($request->body) && is_array($request->body) ? $request->body : [];
                } 
                // Check for direct method call structure
                elseif (isset($request->method)) {
                    $method = $request->method;
                    $params = isset($request->params) && is_array($request->params) ? $request->params : [];
                }
                // Check for array-like structure
                elseif (is_array($request) && count($request) > 0) {
                    $first = $request[0];
                    if (is_string($first)) {
                        $service = $first;
                        if (isset($request[1])) $method = $request[1];
                        if (isset($request[2])) $params = $request[2];
                    }
                }
            } elseif (is_array($request) && count($request) > 0) {
                // Array format: [service, method, params]
                if (isset($request[0])) $service = $request[0];
                if (isset($request[1])) $method = $request[1];
                if (isset($request[2])) $params = $request[2];
            }
            
            error_log("Parsed: service=$service, method=$method, params=" . count($params));
        } catch (Exception $e) {
            error_log("AMF Deserialize Error: " . $e->getMessage());
            // Use defaults
        }
    }
    
    // Handle request
    $result = $handler->handle($service, $method, $params);
    
    // Format response sesuai format Ninja Saga
    // ZendAMF client expects: {status, result, signature, ...}
    $response = new stdClass();
    $response->status = '1';
    $response->result = $result;
    $response->signature = md5(json_encode($result) . APP_SECRET);
    $response->account_lock = null;
    $response->country_area = null;
    $response->isNewAccount = null;
    $response->promote_id = null;
    $response->isTrialEmblem = 0;
    $response->isExpired = 0;
    
    // Serialize AMF response
    // For ZendAMF, we need to wrap in AMF message format
    try {
        // Try to serialize as AMF message
        // ZendAMF expects: AMF message with headers and body
        $amfResponse = $amf->serialize($response);
        
        error_log("AMF Response size: " . strlen($amfResponse) . " bytes");
        
        // Output response
        echo $amfResponse;
        
    } catch (Exception $e) {
        error_log("AMF Serialize Error: " . $e->getMessage());
        error_log("Stack: " . $e->getTraceAsString());
        
        // Last resort: try JSON (won't work but helps debug)
        header('Content-Type: application/json');
        echo json_encode([
            'status' => '0',
            'error' => 'AMF Serialize failed: ' . $e->getMessage(),
            'debug_response' => $response
        ]);
    }
    
} catch (Exception $e) {
    error_log("AMF Gateway Error: " . $e->getMessage());
    error_log("Stack: " . $e->getTraceAsString());
    
    // Error response
    $errorResponse = new stdClass();
    $errorResponse->status = '0';
    $errorResponse->error = $e->getMessage();
    
    try {
        $amf = new AMF();
        $amf->setSerializer(new Serializer());
        $amfError = $amf->serialize($errorResponse);
        echo $amfError;
    } catch (Exception $e2) {
        header('Content-Type: application/json');
        echo json_encode($errorResponse);
    }
}
?>

