<?php

class BattleSystem
{
    /** @var PDO */
    private $db;

    public function __construct($db)
    {
        $this->db = $db;
    }

    /**
     * Helper: validate session and load basic character row.
     */
    private function loadCharacter($charId, $sessionKey)
    {
        $account = validateSessionKey($sessionKey);
        if (!$account) {
            throw new Exception('Invalid session');
        }

        $stmt = $this->db->prepare('
            SELECT character_id, character_level, character_xp, character_gold
            FROM characters
            WHERE character_id = ? AND account_id = ?
            LIMIT 1
        ');
        $stmt->execute(array($charId, $account['account_id']));
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$row) {
            throw new Exception('Character not found');
        }

        return $row;
    }

    /**
     * Start a normal / Grade-S mission.
     *
     * Called as:
     *   BattleSystem.startMission(
     *       [char_id, mission_id, enemyIdsStr, enemyStatsStr, agilityStr, hash, sessionKey]
     *   );
     * or (Grade S):
     *   BattleSystem.startMission(
     *       [char_id, mission_id, enemyIdsStr, enemyStatsStr, agilityStr, hash, sessionKey, stage_grade_s_mission]
     *   );
     *
     * Client expects a 10-character "battle code" (string or array-like with length 10).
     */
    public function startMission()
    {
        $args = func_get_args();

        $charId     = isset($args[0]) ? (int)$args[0] : 0;
        $missionId  = isset($args[1]) ? (string)$args[1] : '';
        // $enemyIds   = isset($args[2]) ? (string)$args[2] : '';
        // $enemyStats = isset($args[3]) ? (string)$args[3] : '';
        // $agility    = isset($args[4]) ? (string)$args[4] : '';
        // $hash       = isset($args[5]) ? (string)$args[5] : '';
        $sessionKey = isset($args[6]) ? (string)$args[6] : '';
        // $stageGrade = isset($args[7]) ? (int)$args[7] : 0; // only for Grade-S mission

        // Validate basic character/session; throw on failure
        $this->loadCharacter($charId, $sessionKey);

        // Generate a pseudo-random 10-character battle code.
        // Client hanya mengecek panjang == 10 sebelum masuk Combat.Battle.
        $code = substr(sha1(uniqid($missionId . '|' . $charId, true)), 0, 10);

        return $code;
    }

    /**
     * Start Sage Scroll mini game (no enemies array).
     * Client expects: { status:1, code: <battle_code> }
     */
    public function startSageScrollMiniGame($charId, $sessionKey, $missionId)
    {
        // Pastikan sesi & karakter valid
        $this->loadCharacter((int)$charId, (string)$sessionKey);

        $code = substr(sha1(uniqid($missionId . '|' . $charId, true)), 0, 10);

        $resp = new stdClass();
        $resp->status = 1;
        $resp->code   = $code;
        return $resp;
    }

    /**
     * Grade-S mission map data.
     * Client (SMissionMap.as) expects:
     *   { status, stage, energy, max_energy, materials:{material_899,material_900} }
     */
    public function getMissionSData($charId, $sessionKey)
    {
        $row = $this->loadCharacter((int)$charId, (string)$sessionKey);

        $resp = new stdClass();
        $resp->status     = 1;
        $resp->stage      = 1;   // unlock stage 1 by default
        $resp->energy     = 100; // simple fixed energy pool
        $resp->max_energy = 100;

        $materials = new stdClass();
        $materials->material_899 = 0;
        $materials->material_900 = 0;
        $resp->materials = $materials;

        return $resp;
    }

    /**
     * Use ramen refill item for Grade-S missions.
     * Params: (char_id, sessionKey, refillType)
     * Client expects same shape as getMissionSData plus result message.
     */
    public function refillRamenMissionS($charId, $sessionKey, $refillType)
    {
        $this->loadCharacter((int)$charId, (string)$sessionKey);

        $resp = new stdClass();
        $resp->status     = 1;
        $resp->energy     = 100;
        $resp->max_energy = 100;

        $materials = new stdClass();
        $materials->material_899 = 0;
        $materials->material_900 = 0;
        $resp->materials = $materials;

        $resp->result = 'Energy refilled';

        return $resp;
    }

    /**
     * Buy ramen items for Grade-S missions.
     * Params: (char_id, sessionKey, buyType, amount)
     * Client expects: { status, materials:{...} }
     */
    public function buyRamenMissionS($charId, $sessionKey, $buyType, $amount)
    {
        $this->loadCharacter((int)$charId, (string)$sessionKey);

        $buyType = (string)$buyType;
        $amount  = (int)$amount;
        if ($amount < 0) {
            $amount = 0;
        }

        $resp = new stdClass();
        $resp->status = 1;

        $materials = new stdClass();
        $materials->material_899 = 0;
        $materials->material_900 = 0;

        if ($buyType === 'material_899') {
            $materials->material_899 = $amount;
        } elseif ($buyType === 'material_900') {
            $materials->material_900 = $amount;
        }

        $resp->materials = $materials;
        return $resp;
    }

    /**
     * Finish mission battle.
     *
     * Called from Combat.Battle.as as:
     *   BattleSystem.finishMission(
     *     [char_id, mission_id, battle_code, hash, totalDamage, sessionKey, packedStatus, stage_grade_s_mission]
     *   )
     *
     * Client expects an object with at least:
     *   status, xp, level, level_up, result [xpReward, goldReward, text]
     */
    public function finishMission()
    {
        $args = func_get_args();

        $charId     = isset($args[0]) ? (int)$args[0] : 0;
        $missionId  = isset($args[1]) ? (string)$args[1] : '';
        // $battleCode = isset($args[2]) ? (string)$args[2] : '';
        // $hash       = isset($args[3]) ? (string)$args[3] : '';
        // $totalDmg   = isset($args[4]) ? (int)$args[4] : 0;
        $sessionKey = isset($args[5]) ? (string)$args[5] : '';
        // $packed     = isset($args[6]) ? (string)$args[6] : '';
        // $stageGrade = isset($args[7]) ? (int)$args[7] : 0;

        $row = $this->loadCharacter($charId, $sessionKey);

        $resp = new stdClass();
        $resp->status   = 1;
        $resp->xp       = (int)$row['character_xp'];
        $resp->level    = (int)$row['character_level'];
        $resp->level_up = false; // sederhana dulu, belum ada perhitungan exp/level up

        // result[0] = xpReward, result[1] = goldReward, result[2] = message
        $resp->result = array(0, 0, 'Mission completed');

        return $resp;
    }
}
