<?php
/**
 * Minimal PetService stub to satisfy PetService.executeService calls
 * from the Flash client. This implementation does NOT implement
 * real pet logic; it only returns safe empty data structures so
 * that the Pet panels can open without errors.
 */

class PetService
{
    /** @var PDO */
    private $db;

    public function __construct($db)
    {
        $this->db = $db;
    }

    /**
     * Helper: load character_inventory JSON as associative array
     */
    private function loadInventory($charId, $accountId)
    {
        $stmt = $this->db->prepare("SELECT character_inventory FROM characters WHERE character_id = ? AND account_id = ? LIMIT 1");
        $stmt->execute(array($charId, $accountId));
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        $inv = array();
        if ($row && !empty($row['character_inventory'])) {
            $decoded = json_decode($row['character_inventory'], true);
            if (is_array($decoded)) {
                $inv = $decoded;
            }
        }
        return $inv;
    }

    /**
     * Helper: save character_inventory JSON
     */
    private function saveInventory($charId, $accountId, array $inv)
    {
        $invJson = json_encode($inv);
        $stmt = $this->db->prepare("UPDATE characters SET character_inventory = ? WHERE character_id = ? AND account_id = ?");
        $stmt->execute(array($invJson, $charId, $accountId));
    }

    /**
     * Helper: build pets[] array from char_pets string
     * char_pets format (very simple for now): "petId1,petId2,petId3"
     */
    private function buildPetsFromInventory(array $inv)
    {
        $pets = array();

        // List kepemilikan pet sederhana: "petId1,petId2,petId3"
        $raw  = isset($inv['char_pets']) ? (string)$inv['char_pets'] : '';
        if ($raw === '') {
            return $pets;
        }

        // Data detail per-pet (level, xp, mp, skills, name) disimpan di key "pet_data"
        $allMeta = array();
        if (isset($inv['pet_data']) && is_array($inv['pet_data'])) {
            $allMeta = $inv['pet_data'];
        }

        $ids = explode(',', $raw);
        $ids = array_values(array_filter($ids, function ($v) {
            return trim($v) !== '';
        }));

        $index = 1;
        foreach ($ids as $pid) {
            $pid = trim($pid);
            if ($pid === '') {
                $index++;
                continue;
            }

            $meta = array();
            if (isset($allMeta[$pid]) && is_array($allMeta[$pid])) {
                $meta = $allMeta[$pid];
            }

            // ------- Normalisasi statistik dasar -------
            $level = isset($meta['level']) ? (int)$meta['level'] : 1;
            if ($level < 1) {
                $level = 1;
            }

            $xp = isset($meta['xp']) ? (int)$meta['xp'] : 0;
            if ($xp < 0) {
                $xp = 0;
            }

            $mp = isset($meta['mp']) ? (int)$meta['mp'] : 100;
            if ($mp < 0) {
                $mp = 0;
            } elseif ($mp > 100) {
                $mp = 100;
            }

            // Skills string "0,0,0,0,0,0" (6 slot, 0=belum learned, 1=learned)
            $skillsStr = isset($meta['skills']) ? (string)$meta['skills'] : '0,0,0,0,0,0';
            $skillsArr = explode(',', $skillsStr);
            $norm      = array();
            for ($i = 0; $i < 6; $i++) {
                $val = isset($skillsArr[$i]) ? (int)$skillsArr[$i] : 0;
                $norm[$i] = ($val > 0) ? 1 : 0;
            }
            $skillsStr = implode(',', $norm);

            // Nama pet: pakai custom name kalau ada, kalau tidak generate dari id
            if (isset($meta['name']) && $meta['name'] !== '') {
                $petName = (string)$meta['name'];
            } else {
                $petName = $pid;
                if (strpos($pid, 'pet_') === 0) {
                    $slug = substr($pid, 4);           // "pet_yamaru" -> "yamaru"
                    $slug = str_replace('_', ' ', $slug);
                    $petName = ucwords($slug);         // "yamaru" -> "Yamaru"
                }
            }

            $p = new stdClass();
            // Basic identifiers: pet_id = index numerik, pet_swf = id string untuk SWF
            $p->pet_id   = $index;    // index 1..N per karakter, dipakai panel untuk equip/unequip
            $p->pet_swf  = $pid;      // SWF name, dipakai PetInfo di client
            $p->pet_name = $petName;

            // Stats
            $p->pet_level = $level;
            $p->pet_xp    = $xp;
            $p->pet_mp    = $mp;
            $p->pet_skills = $skillsStr;

            $pets[] = $p;

            $index++;
        }

        return $pets;
    }

    /**
     * Generic entry point used by the client:
     * PetService.executeService(command, paramsArray)
     *
     * We support at least the commands used in Pets.as / PetVilla.as /
     * PetShop.as / PetInventory.as:
     *  - "getPets"
     *  - "getVillaData"
     *  - "buyPet"
     *  - "learnSkill"
     *  - "feed"
     *  - "setLevel"
     *  - "releasePet"
     *  - "equipPet"
     *  - "unequipPet"
     *  - "unlockSlots"
     *  - "trainPet"
     *  - "checkoutPet"
     *  - "skipTraining"
     *  - "cancelTraining"
     *
     * For now everything returns status=1 and minimal structures.
     */
    public function executeService($command, $params)
    {
        $result = new stdClass();
        $result->status = 1;

        switch ($command) {
            case 'getPets':
                // params: [0] = char_id, [1] = sessionKey
                $charId     = isset($params[0]) ? (int)$params[0] : 0;
                $sessionKey = isset($params[1]) ? (string)$params[1] : '';

                $account = validateSessionKey($sessionKey);
                if (!$account) {
                    $result->status = 0;
                    $result->error  = 'Invalid session';
                    break;
                }

                $inv        = $this->loadInventory($charId, $account['account_id']);
                $result->pets = $this->buildPetsFromInventory($inv);
                break;

            case 'getVillaData':
                // params: [0] = char_id, [1] = sessionKey
                $charId     = isset($params[0]) ? (int)$params[0] : 0;
                $sessionKey = isset($params[1]) ? (string)$params[1] : '';

                $account = validateSessionKey($sessionKey);
                if (!$account) {
                    $result->status = 0;
                    $result->error  = 'Invalid session';
                    break;
                }

                $inv         = $this->loadInventory($charId, $account['account_id']);
                $result->pets = $this->buildPetsFromInventory($inv);

                // Villa training slots: for now, just 4 slots all locked (status=0)
                $result->slots = array();
                for ($i = 0; $i < 4; $i++) {
                    $slot = new stdClass();
                    $slot->status      = 0;    // 0=locked, 1=empty, 2=training, 3=done (see PetVilla.as)
                    $slot->pet_id      = 0;
                    $slot->skip_price  = 0;
                    $slot->completed_at= 0;
                    $result->slots[]   = $slot;
                }
                break;

            case 'buyPet':
                // params: [0] = char_id, [1] = sessionKey, [2] = petId
                $charId     = isset($params[0]) ? (int)$params[0] : 0;
                $sessionKey = isset($params[1]) ? (string)$params[1] : '';
                $petId      = isset($params[2]) ? (string)$params[2] : '';

                $account = validateSessionKey($sessionKey);
                if (!$account) {
                    $result->status = 0;
                    $result->error  = 'Invalid session';
                    break;
                }

                if ($petId === '') {
                    $result->status = 0;
                    $result->error  = 'Invalid pet';
                    break;
                }

                $inv = $this->loadInventory($charId, $account['account_id']);
                $raw = isset($inv['char_pets']) ? (string)$inv['char_pets'] : '';
                $ids = $raw !== '' ? explode(',', $raw) : array();

                if (!in_array($petId, $ids, true)) {
                    $ids[] = $petId;
                }

                // Bersihkan elemen kosong lalu simpan kembali
                $ids = array_filter($ids, function ($v) {
                    return trim($v) !== '';
                });
                $inv['char_pets'] = implode(',', $ids);

                // Inisialisasi data pet default di inventory (level/xp/mp/skills/name)
                if (!isset($inv['pet_data']) || !is_array($inv['pet_data'])) {
                    $inv['pet_data'] = array();
                }
                if (!isset($inv['pet_data'][$petId]) || !is_array($inv['pet_data'][$petId])) {
                    $inv['pet_data'][$petId] = array(
                        'level'  => 1,
                        'xp'     => 0,
                        'mp'     => 100,
                        'skills' => '0,0,0,0,0,0',
                        'name'   => '',
                    );
                }

                $this->saveInventory($charId, $account['account_id'], $inv);

                // Biarkan client yang mengurangi gold/token secara lokal, seperti buySkill
                $result->result = 'OK';
                break;

            case 'learnSkill':
                // params: [char_id, sessionKey, petIndex(1-..), skillIndex(1-6), learn_method]
                $charId     = isset($params[0]) ? (int)$params[0] : 0;
                $sessionKey = isset($params[1]) ? (string)$params[1] : '';
                $petIndex   = isset($params[2]) ? (int)$params[2] : 0;
                $skillIndex = isset($params[3]) ? (int)$params[3] : 0; // 1..6

                $account = validateSessionKey($sessionKey);
                if (!$account) {
                    $result->status = 0;
                    $result->error  = 'Invalid session';
                    break;
                }

                if ($charId <= 0 || $petIndex <= 0 || $skillIndex < 1 || $skillIndex > 6) {
                    $result->status = 0;
                    $result->error  = 'Invalid parameters';
                    break;
                }

                $inv = $this->loadInventory($charId, $account['account_id']);

                // Pastikan pet dimiliki (petIndex -> id string di char_pets)
                $raw = isset($inv['char_pets']) ? (string)$inv['char_pets'] : '';
                $ids = $raw !== '' ? explode(',', $raw) : array();
                $ids = array_values(array_filter($ids, function ($v) {
                    return trim($v) !== '';
                }));

                if (!isset($ids[$petIndex - 1])) {
                    $result->status = 2;
                    $result->result = 'Pet not owned';
                    break;
                }

                $petId = trim($ids[$petIndex - 1]);

                if (!isset($inv['pet_data']) || !is_array($inv['pet_data'])) {
                    $inv['pet_data'] = array();
                }
                if (!isset($inv['pet_data'][$petId]) || !is_array($inv['pet_data'][$petId])) {
                    $inv['pet_data'][$petId] = array(
                        'level'  => 1,
                        'xp'     => 0,
                        'mp'     => 100,
                        'skills' => '0,0,0,0,0,0',
                        'name'   => '',
                    );
                }

                $skillsStr = isset($inv['pet_data'][$petId]['skills']) ? (string)$inv['pet_data'][$petId]['skills'] : '0,0,0,0,0,0';
                $skillsArr = explode(',', $skillsStr);
                for ($i = 0; $i < 6; $i++) {
                    if (!isset($skillsArr[$i]) || trim($skillsArr[$i]) === '') {
                        $skillsArr[$i] = 0;
                    } else {
                        $skillsArr[$i] = (int)$skillsArr[$i];
                    }
                }
                $idx = $skillIndex - 1;
                $skillsArr[$idx] = 1; // tandai skill learned
                $inv['pet_data'][$petId]['skills'] = implode(',', $skillsArr);

                $this->saveInventory($charId, $account['account_id'], $inv);
                $result->result = 'OK';
                break;

            case 'feed':
                // params: [char_id, sessionKey, petIndex(1-..), foodId]
                $charId     = isset($params[0]) ? (int)$params[0] : 0;
                $sessionKey = isset($params[1]) ? (string)$params[1] : '';
                $petIndex   = isset($params[2]) ? (int)$params[2] : 0;

                $account = validateSessionKey($sessionKey);
                if (!$account) {
                    $result->status = 0;
                    $result->error  = 'Invalid session';
                    break;
                }

                if ($charId <= 0 || $petIndex <= 0) {
                    $result->status = 0;
                    $result->error  = 'Invalid parameters';
                    break;
                }

                $inv = $this->loadInventory($charId, $account['account_id']);

                $raw = isset($inv['char_pets']) ? (string)$inv['char_pets'] : '';
                $ids = $raw !== '' ? explode(',', $raw) : array();
                $ids = array_values(array_filter($ids, function ($v) {
                    return trim($v) !== '';
                }));

                if (!isset($ids[$petIndex - 1])) {
                    $result->status = 2;
                    $result->result = 'Pet not owned';
                    break;
                }

                $petId = trim($ids[$petIndex - 1]);

                if (!isset($inv['pet_data']) || !is_array($inv['pet_data']) || !isset($inv['pet_data'][$petId])) {
                    $result->status = 2;
                    $result->result = 'Pet data not found';
                    break;
                }

                $meta = $inv['pet_data'][$petId];
                $mp   = isset($meta['mp']) ? (int)$meta['mp'] : 100;
                $mp  += 10; // setiap feed +10 MP, max 100
                if ($mp > 100) {
                    $mp = 100;
                }
                if ($mp < 0) {
                    $mp = 0;
                }

                $inv['pet_data'][$petId]['mp'] = $mp;
                $this->saveInventory($charId, $account['account_id'], $inv);

                $result->mp = $mp;
                $result->result = 'OK';
                break;

            case 'setLevel':
                // params: [char_id, sessionKey, petIndex(1-..)]
                $charId     = isset($params[0]) ? (int)$params[0] : 0;
                $sessionKey = isset($params[1]) ? (string)$params[1] : '';
                $petIndex   = isset($params[2]) ? (int)$params[2] : 0;

                $account = validateSessionKey($sessionKey);
                if (!$account) {
                    $result->status = 0;
                    $result->error  = 'Invalid session';
                    break;
                }

                if ($charId <= 0 || $petId === '') {
                    $result->status = 0;
                    $result->error  = 'Invalid parameters';
                    break;
                }

                $inv = $this->loadInventory($charId, $account['account_id']);
                if (!isset($inv['pet_data']) || !is_array($inv['pet_data']) || !isset($inv['pet_data'][$petId])) {
                    $result->status = 2;
                    $result->result = 'Pet data not found';
                    break;
                }

                // Naikkan ke level tinggi (misal 60) dan unlock semua skill
                $inv['pet_data'][$petId]['level']  = 60;
                $inv['pet_data'][$petId]['xp']     = 0;
                $inv['pet_data'][$petId]['mp']     = 100;
                $inv['pet_data'][$petId]['skills'] = '1,1,1,1,1,1';

                $this->saveInventory($charId, $account['account_id'], $inv);

                $result->result = 'OK';
                break;

            case 'renamePet':
                // params: [char_id, sessionKey, petIndex(1-..), newName]
                $charId     = isset($params[0]) ? (int)$params[0] : 0;
                $sessionKey = isset($params[1]) ? (string)$params[1] : '';
                $petIndex   = isset($params[2]) ? (int)$params[2] : 0;
                $newName    = isset($params[3]) ? (string)$params[3] : '';

                $account = validateSessionKey($sessionKey);
                if (!$account) {
                    $result->status = 0;
                    $result->error  = 'Invalid session';
                    break;
                }

                if ($charId <= 0 || $petIndex <= 0) {
                    $result->status = 0;
                    $result->error  = 'Invalid parameters';
                    break;
                }

                $inv = $this->loadInventory($charId, $account['account_id']);

                // Pastikan pet dimiliki sebelum di-rename (petIndex -> id string di char_pets)
                $raw = isset($inv['char_pets']) ? (string)$inv['char_pets'] : '';
                $ids = $raw !== '' ? explode(',', $raw) : array();
                $ids = array_values(array_filter($ids, function ($v) {
                    return trim($v) !== '';
                }));

                if (!isset($ids[$petIndex - 1])) {
                    $result->status = 2;
                    $result->result = 'Pet not owned';
                    break;
                }

                $petId = trim($ids[$petIndex - 1]);

                if (!isset($inv['pet_data']) || !is_array($inv['pet_data'])) {
                    $inv['pet_data'] = array();
                }
                if (!isset($inv['pet_data'][$petId]) || !is_array($inv['pet_data'][$petId])) {
                    $inv['pet_data'][$petId] = array(
                        'level'  => 1,
                        'xp'     => 0,
                        'mp'     => 100,
                        'skills' => '0,0,0,0,0,0',
                        'name'   => '',
                    );
                }

                // Simpan nama baru (boleh string kosong untuk reset ke default dari id)
                $inv['pet_data'][$petId]['name'] = $newName;

                $this->saveInventory($charId, $account['account_id'], $inv);

                $result->result   = 'OK';
                $result->pet_id   = $petIndex;
                $result->pet_name = $newName;
                break;

            case 'releasePet':
                // params: [char_id, sessionKey, petIndex(1-..)]
                $charId     = isset($params[0]) ? (int)$params[0] : 0;
                $sessionKey = isset($params[1]) ? (string)$params[1] : '';
                $petIndex   = isset($params[2]) ? (int)$params[2] : 0;

                $account = validateSessionKey($sessionKey);
                if (!$account) {
                    $result->status = 0;
                    $result->error  = 'Invalid session';
                    break;
                }

                if ($charId <= 0 || $petIndex <= 0) {
                    $result->status = 0;
                    $result->error  = 'Invalid parameters';
                    break;
                }

                $inv = $this->loadInventory($charId, $account['account_id']);
                $raw = isset($inv['char_pets']) ? (string)$inv['char_pets'] : '';
                $ids = $raw !== '' ? explode(',', $raw) : array();
                $ids = array_values(array_filter($ids, function ($v) {
                    return trim($v) !== '';
                }));

                if (!isset($ids[$petIndex - 1])) {
                    $result->status = 2;
                    $result->result = 'Pet not owned';
                    break;
                }

                $petId = trim($ids[$petIndex - 1]);

                // Hapus pet dari daftar dan data detail
                unset($ids[$petIndex - 1]);
                $inv['char_pets'] = implode(',', $ids);

                if (isset($inv['pet_data'][$petId])) {
                    unset($inv['pet_data'][$petId]);
                }

                $this->saveInventory($charId, $account['account_id'], $inv);

                // Jika pet yang di-release sedang di-equip, kosongkan di tabel characters
                try {
                    $stmt = $this->db->prepare("UPDATE characters SET character_pet = '', character_pet_id = 0 WHERE character_id = ? AND account_id = ? AND character_pet = ?");
                    $stmt->execute(array($charId, $account['account_id'], $petId));
                } catch (Exception $e) {
                    // abaikan error, tidak kritis
                }

                $result->result = 'OK';
                break;

            case 'equipPet':
                // params: [char_id, sessionKey, petIndex(1-..)]
                $charId     = isset($params[0]) ? (int)$params[0] : 0;
                $sessionKey = isset($params[1]) ? (string)$params[1] : '';
                $petIndex   = isset($params[2]) ? (int)$params[2] : 0;

                $account = validateSessionKey($sessionKey);
                if (!$account) {
                    $result->status = 0;
                    $result->error  = 'Invalid session';
                    break;
                }

                if ($charId <= 0 || $petIndex <= 0) {
                    $result->status = 0;
                    $result->error  = 'Invalid parameters';
                    break;
                }

                // Pastikan pet dimiliki (petIndex -> id string di char_pets)
                $inv = $this->loadInventory($charId, $account['account_id']);
                $raw = isset($inv['char_pets']) ? (string)$inv['char_pets'] : '';
                $ids = $raw !== '' ? explode(',', $raw) : array();
                $ids = array_values(array_filter($ids, function ($v) {
                    return trim($v) !== '';
                }));

                if (!isset($ids[$petIndex - 1])) {
                    $result->status = 2;
                    $result->result = 'Pet not owned';
                    break;
                }

                $petId = trim($ids[$petIndex - 1]);

                // Simpan equip di tabel characters
                $stmt = $this->db->prepare("UPDATE characters SET character_pet = ?, character_pet_id = ? WHERE character_id = ? AND account_id = ?");
                $stmt->execute(array($petId, $petIndex, $charId, $account['account_id']));

                $result->pet_id  = $petIndex;
                $result->pet_swf = $petId;
                $result->result  = 'OK';
                break;

            case 'unequipPet':
                // params: [char_id, sessionKey]
                $charId     = isset($params[0]) ? (int)$params[0] : 0;
                $sessionKey = isset($params[1]) ? (string)$params[1] : '';

                $account = validateSessionKey($sessionKey);
                if (!$account) {
                    $result->status = 0;
                    $result->error  = 'Invalid session';
                    break;
                }

                if ($charId <= 0) {
                    $result->status = 0;
                    $result->error  = 'Invalid parameters';
                    break;
                }

                $stmt = $this->db->prepare("UPDATE characters SET character_pet = '', character_pet_id = 0 WHERE character_id = ? AND account_id = ?");
                $stmt->execute(array($charId, $account['account_id']));

                $result->result  = 'OK';
                $result->pet_id  = 0;
                $result->pet_swf = '';
                break;

            case 'unlockSlots':
            case 'trainPet':
            case 'checkoutPet':
            case 'skipTraining':
            case 'cancelTraining':
                // Untuk sekarang belum ada sistem training penuh, cukup OK supaya UI tidak error.
                $result->result = 'OK';
                break;

            default:
                // Unknown command: still do not fail hard, just report status>1
                $result->status = 2;
                $result->result = 'Unknown pet command';
                break;
        }

        return $result;
    }
}
